/* 
 * The "withpass" utility provides the user with an alternative 
 * password input method for use in applications that expect a 
 * cleartext password as a command line argument.
 * 
 * The user may provide the password value either at the interactive prompt,
 * or as an environmental variable (WITHPASS). The latter approach comes with 
 * some security considerations but is nonetheless useful for shell scripts.
 * 
 * You can use "withpass" to avoid recording passwords in the logs of 
 * LD_PRELOAD-based execve() loggers (such as "snoopy").
 * 
 * Build: gcc -Wall -static -o withpass withpass.c
 * 
 * Usage: ./withpass <cmd> <arg1> ... <argN>
 *
 * Examples: ./withpass net rpc shutdown -t 0 -I 192.168.0.1 -Uuser%%P
 *           ./withpass wget --http-user u1 --http-password %P http://u.rl/1
 *
 * Note: Each argument may contain only one %P
 *
 * (c) Dimitris Glynos < dimitris { at } census-labs.com >
 *
 * Licensed under GPLv3 (see LICENSE file)
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

/*
 * Replaces the first instance of %P on each cmdline argument
 * with the user supplied password.
 */

char **replace_pass(int argc, char **cmd, const char *pass)
{
	char *copied, *found;
	int i, copied_len, pass_len;
	
	pass_len = strlen(pass);

	cmd[0] = cmd[1];

	for (i=2; i < argc; i++) {
		found = strstr(cmd[i], "%P");

		if (!found)
			continue;

		copied_len = strlen(cmd[i]) - 2 + pass_len + 1;
		copied = malloc(copied_len);
		copied[copied_len - 1] = '\0';
		*found = '\0';

		snprintf(copied, copied_len, "%s%s%s", cmd[i], pass, found+2);
		cmd[i] = copied;
	}
	return cmd;
}

int main(int argc, char *argv[])
{
	char *pass, *pass_env;
	int ret;

	if (argc < 3) {
		fprintf(stderr, "usage: %s <cmd> <arg1> ... <argN>\n", argv[0]);
		exit(1);
	}

	pass_env = getenv("WITHPASS");

	if (pass_env) {
		pass = strdup(pass_env);
		memset(pass_env, 'X', strlen(pass_env));
		unsetenv("WITHPASS");
	} else {
		pass = getpass("<withpass> enter password: ");
	}

	argv = replace_pass(argc, argv, pass);
	ret = execvp(argv[0], &argv[1]);

	if (ret == -1) {
		fprintf(stderr, "error while executing cmd %s: %s\n",
			argv[0], strerror(errno));
		exit(1);
	}

	return 0;
}
